from knot_utilities import IsKnot, NPlusMinus
from collections import defaultdict

####### KHOVANOV HOMOLOGY #######

#### Includes the following functions:

## KhovanovComplex(K, Max_Crossing_Number = 14)


def KhovanovComplex(K, Max_Crossing_Number = 14):
    """ 
    Returns the Khovanov complex for a projection K with no more than 14 crossings.

    The return value is (List1, List2), where 
    List1 gives the generators of the Khovanov complex, and 
    List2 gives the non-zero entries in the boundary map.

    List1 = [(a_1,b_1),...,(a_k,b_k)], 
    where (a_i,b_i) corresponds to a Khovanov generator with: 
    homological grading = a_i, q-grading = b_i.

    List2 = [(p_1,q_1,e_1),...,(p_n,q_n,e_n)], where
    0 <= p_i < len(List1), 0 <= q_i < len(List2), e_i = \pm 1.
    A term (p,q,e) means that in the boundary List1[p], 
    the generator List1[q] appears with coefficient e.
    """
    
    if IsKnot(K) == False:
        print("Not a knot")
        return
    
    GeneratorList = []
    ArrowList = []
    n_plus, n_minus = NPlusMinus(K)
    n_crossings = n_plus + n_minus

    if n_crossings > Max_Crossing_Number:
        print("Number of Crossings > ", Max_Crossing_Number)
        print("Terminating Computation.")
        return
              
    GeneratorList.append(((1,0),-n_minus, n_plus-2*n_minus)) # We take care of the overall shifts at the beginning.

    # The generators G in GeneratorList  are represented by (P, a, b).

    # P is a special fixed-point free involution on (0,1,...,2n-1),
    # P[i] != i, P[[i]] = i
    # with the property that there are no  x < y < z < w with P[x] = z and P[y] = w. (no linked intervals).
    # Such an involution is called a "crossingless matching".

    # Examples when n = 2: (3,2,1,0), (1,0,3,2).
    # Examples when n = 3: (1,0,3,2,5,4), (1,0,5,4,3,2), (3,2,1,0,5,4), (5,2,1,4,3,0), (5,4,3,2,1,0).

    # a and b are integers. a is the homological grading, b is the q grading.

    # The arrows Arrow in ArrowList will be represented by (Head, Tail, Element, Sign).
    # Head and Tail are integers, Sign = \pm 1.
    # The arrow goes from GeneratorsList[Head] to GeneratorsList[Tail].

    # Element is an algebra element.
    # In the current case Algebra can be:
    
    # 1, represented by ()
    # Multiplication by x on the arc that connects s1 and s2, represented by (s1,s2)
    # A cobordism between arcs (s1,s2) and (s3,s4), represented by (s1,s2,s3,s4).
    # (Note that at this point we don't know whether this will be a join or a split cobordism). 
 
    # If P1 and P2 denote the crossingless matchings of the head and tail generators:
    # In the first two cases P1 = P2. 
    # In the last case P1[s1] = s2, P1[s3] = s4.
    # We also have the constraints that
    # P1[j] = P2[j] for j not in (p1, p2, p3, p4),
    # P2[s1] in {s3,s4},
    # P2[s2] in {s3,s4}.
    # (Since P2 is a crossingless matching, these constraints uniquely determine P2).
    
    # When (s1,s2) = (s3,s4) it represents a dotted cobordism.  
    # Otherwise it represensents a surgery arc from (s1,s2) to (s3,s4).
    
    
    for i in range(1,len(K)):
        c = K[i]
        NewGeneratorList = []
        NewArrowList = []
        
        if c > 100:   # Maximum. This is the easiest case. Each generator gives exactly one new generator.
            s = c-101
            # Note that 0 <= s <= 2n-2, where the horizontal line after
            # passing through the maximum intersects the knot in 2n points.

            # First update the generator list:
            for G in GeneratorList:
                P, a, b = G  
                new_P = [p if p < s else p+2 for p in P[:s]] + [s+1,s] + [p if p < s else p+2 for p in P[s:]]
                new_P = tuple(new_P) 
                new_G = (new_P, a, b)
                NewGeneratorList.append(new_G)

            # Then update the differential:
            for Arrow in ArrowList:
                Head, Tail, Element, Sign = Arrow
                new_Element = tuple([p if p < s else p+2 for p in Element])
                new_Arrow = (Head, Tail, new_Element, Sign)
                NewArrowList.append(new_Arrow)
                

        elif c < -100:  # Minimum. Generators can give 1 or 2 new generators.
            s = -c-101

            # First update the generators:
            Table1 = {} # Lookup table for generators that have 1 extension.
            Table2 = {} # Lookup table for generators that have 2 extensions.
            index = 0   # The index of the new generator that we create in the NewGeneratorList.
            for j in range(len(GeneratorList)):
                P, a, b = GeneratorList[j]
                s1 = P[s]
                s2 = P[s+1]
                if s1 != s+1:
                    new_P = list(P)
                    new_P[s1] = s2
                    new_P[s2] = s1
                    new_P = [p if p < s else p-2 for p in new_P[:s]] + [p if p < s else p-2 for p in new_P[s+2:]]
                    new_P = tuple(new_P)
                    new_G = (new_P, a, b)
                    NewGeneratorList.append(new_G)
                    Table1[j] = index # equals to len(NewGeneratorList)-1
                    index+=1
                   
                else : # we create a new circle. This gives two generators.
                    new_P = [p if p < s else p-2 for p in P[:s]] + [p if p < s else p-2 for p in P[s+2:]]
                    new_P = tuple(new_P)
                    new_G1 = (new_P, a, b+1) # corresponds to v+ on the new circle
                    new_G2 = (new_P, a, b-1) # corresponds to v- on the new circle 
                    NewGeneratorList.append(new_G1)
                    NewGeneratorList.append(new_G2)
                    Table2[j] = index
                    index +=2 # Since we have just created 2 new generators.

            # Then add the new Arrows:
            for Arrow in ArrowList:
                Head, Tail, Element, Sign = Arrow
                if Head in Table1 and Tail in Table1: 
                    new_Head = Table1[Head]
                    new_Tail = Table1[Tail]
                    P, _, _ = GeneratorList[Head]
                    new_Element = []
                    for p in Element:
                        if p == s:
                            new_Element.append(P[s+1])
                        elif p == s+1:
                            new_Element.append(P[s])
                        else:
                            new_Element.append(p)
                    new_Element = [p if p < s else p-2 for p in new_Element]
                    new_Element = tuple(new_Element)
                    new_Arrow = (new_Head, new_Tail, new_Element, Sign)
                    NewArrowList.append(new_Arrow)
                    
                if Head in Table1 and Tail in Table2:
                    # This means that Element is a cobordism. After surgery we get the (s,s+1) arc.
                    # This means that before the minimum P1[s]=y, P1[s+1]=z, P2[y]=z, P2[s]=s+1.
                    # After the minimum new_P[y'] = z' for both generators
                    # This is now a split cobordism:
                    # if we use v- on the new circle, the algebra element is 1,
                    # if we use v+ on the new circle, the algebra element is multiplication by x, represented by (y',z').
                    
                    new_Head = Table1[Head]
                    new_Tail  = Table2[Tail] # Using v+.
                    P1, _, _ = GeneratorList[Head]
                    new_arc = [P1[s],P1[s+1]]
                    new_arc = [p if p < s else p-2 for p in new_arc]
                    new_arc = tuple(new_arc)
                    
                    new_Arrow1 = (new_Head, new_Tail  , new_arc, Sign)  # Using v+ on the new circle.
                    new_Arrow2 = (new_Head, new_Tail+1, ()     , Sign)  # Using v- on the new circle.
                    NewArrowList+=[new_Arrow1, new_Arrow2]

                if Head in Table2 and Tail in Table1:  # In this case we have a join map.
                    new_Head = Table2[Head] # Using v+.
                    new_Tail = Table1[Tail]
                    P2, _, _ = GeneratorList[Tail]
                    new_arc = [P2[s],P2[s+1]]
                    new_arc = [p if p < s else p-2 for p in new_arc]
                    new_arc = tuple(new_arc)
                    
                    new_Arrow1 = (new_Head  , new_Tail, ()     , Sign) # Using v+ on the new circle.
                    new_Arrow2 = (new_Head+1, new_Tail, new_arc, Sign) # Using v- on the new circle.
                    NewArrowList+=[new_Arrow1, new_Arrow2]

                if Head in Table2 and Tail in Table2:
                    new_Head = Table2[Head]
                    new_Tail = Table2[Tail]
                    if len(Element) == 2 and s in Element: # In this case Element = (s,s+1) or (s+1,s).
                        new_Arrow = (new_Head, new_Tail+1, (), Sign)
                        NewArrowList.append(new_Arrow)
                        
                    else: 
                        new_Element = [p if p < s else p-2 for p in Element]
                        new_Element = tuple(new_Element)
                        new_Arrow1 =  (new_Head , new_Tail  , new_Element, Sign)
                        new_Arrow2  = (new_Head+1,new_Tail+1, new_Element, Sign)
                        NewArrowList+=[new_Arrow1, new_Arrow2]

            
        else :   # Crossing.
            s = abs(c)-1
            if c > 0:
                shift1 = 0
                shift2 = 1
            else:
                shift1 = 1
                shift2 = 0
                        # First update the generators:
            Table0 = {} # Lookup table for generators that correspond to the vertical resolution.
            Table1 = {} # Lookup table for generators for the other resolution with 1 extension.
            Table2 = {} # Lookup table for generators for the other resolutiin with 2 extensions.
            index = 0   # The index of the new generator that we create in the NewGeneratorList.

            # First we construct the new generators:
            
            for j in range(len(GeneratorList)):
                P, a, b = GeneratorList[j]

                # Adding the vertical resolution:
                new_G = (P, a+shift1, b+shift1)
                NewGeneratorList.append(new_G) 
                Table0[j] = index
                index += 1

                # Adding the other resolution:
                s1 = P[s]
                s2 = P[s+1]
                if s1 != s+1:
                    new_P = list(P)
                    new_P[s1] = s2
                    new_P[s2] = s1
                    new_P = new_P[:s] + [s+1,s]+ new_P[s+2:]
                    new_P = tuple(new_P)
                    new_G = (new_P, a+shift2, b+shift2)
                    NewGeneratorList.append(new_G)
                    Table1[j] = index # Equals to len(NewGeneratorList)-1.
                    index+=1
                   
                else : # We create a new circle. This gives two generators. P is unchanged.
                    new_G1 = (P, a + shift2, b + shift2 +1) # Corresponds to v+ on the new circle.
                    new_G2 = (P, a+  shift2, b + shift2 -1) # Corresponds to v- on the new circle. 
                    NewGeneratorList.append(new_G1)
                    NewGeneratorList.append(new_G2)
                    Table2[j] = index
                    index += 2

            # Then we add the arrows.        
            # We will have 3 kinds of arrows:
            # Arrows for the vertical resolution,
            # Arrows for the other resolution,
            # Connecting arrows between the resolutions.

            # Arrows for the vertical resolution:
            for Arrow in ArrowList:
                Head, Tail, Element, Sign = Arrow
                new_Arrow = (Table0[Head], Table0[Tail], Element, Sign)
                NewArrowList.append(new_Arrow)

            # Arrows for the other resolution:
            for Arrow in ArrowList:
                Head, Tail, Element, Sign = Arrow
                if Head in Table1 and Tail in Table1: 
                    new_Head = Table1[Head]
                    new_Tail = Table1[Tail]
                    P, _, _ = GeneratorList[Head]
                    new_Element = []
                    for p in Element:
                        if p == s:
                            new_Element.append(P[s+1])
                        elif p == s+1:
                            new_Element.append(P[s])
                        else:
                            new_Element.append(p)
                    new_Element = tuple(new_Element)
                    new_Arrow = (new_Head, new_Tail, new_Element, Sign)
                    NewArrowList.append(new_Arrow)
                    
                if Head in Table1 and Tail in Table2:
                    new_Head = Table1[Head]
                    new_Tail  = Table2[Tail] # Using v+.
                    P1, _, _ = GeneratorList[Head]
                    new_arc = [P1[s],P1[s+1]]
                    new_arc = tuple(new_arc)
                    new_Arrow1 = (new_Head, new_Tail  , new_arc, Sign)  # Using v+ on the new circle.
                    new_Arrow2 = (new_Head, new_Tail+1, ()     , Sign)  # Using v- on the new circle.
                    NewArrowList+=[new_Arrow1, new_Arrow2]

                if Head in Table2 and Tail in Table1:  # In this case we have a join map.
                    new_Head = Table2[Head] # Using v+.
                    new_Tail = Table1[Tail]
                    P2, _, _ = GeneratorList[Tail]
                    new_arc = [P2[s],P2[s+1]]
                    new_arc = tuple(new_arc)
                    
                    new_Arrow1 = (new_Head  , new_Tail, ()     , Sign) # Using v+ on the new circle.
                    new_Arrow2 = (new_Head+1, new_Tail, new_arc, Sign) # Using v- on the new circle.
                    NewArrowList+=[new_Arrow1, new_Arrow2]

                if Head in Table2 and Tail in Table2:
                    new_Head = Table2[Head]
                    new_Tail = Table2[Tail]
                    if len(Element) == 2 and s in Element: # In this case Element = (s,s+1) or (s+1,s).
                        new_Arrow = (new_Head, new_Tail+1, (), Sign)
                        NewArrowList.append(new_Arrow)
                        
                    else: 
                        new_Arrow1 =  (new_Head , new_Tail  , Element, Sign)
                        new_Arrow2  = (new_Head+1,new_Tail+1, Element, Sign)
                        NewArrowList+=[new_Arrow1, new_Arrow2]
  

            # Arrows between the resolutions:
            if c > 0: # Arrows go from vertical resolution to the other resolution.
                for i in range(len(GeneratorList)):
                    P, a, _ = GeneratorList[i]
                    # a = partial homological grading.
                    # a = -n_minus + (number of times we used the 1 resolution for the previous crossings)
                    if a%2 == 0:
                        Sign =  1
                    else :
                        Sign = -1
                    if i in Table1:
                        Head = Table0[i]
                        Tail = Table1[i]
                        s1 = P[s]
                        s2 = P[s+1]
                        Element = (s1,s,s+1,s2)
                        new_Arrow = (Head, Tail, Element, Sign)
                        NewArrowList.append(new_Arrow)

                    else : # We have a split map.
                        Head = Table0[i]
                        Tail = Table2[i]
                        new_Arrow1 = (Head, Tail  , (s,s+1), Sign) # Maps to (v+)\otimes x.
                        new_Arrow2 = (Head, Tail+1, ()     , Sign) # Maps to (v-)\otimes 1.
                        NewArrowList+=[new_Arrow1, new_Arrow2]

            else : # Arrows go from the other resolution to the vertical resolution.
                for i in range(len(GeneratorList)):
                    P, a, _ = GeneratorList[i]
                    if a%2 == 0:
                        Sign = 1
                    else :
                        Sign = -1
                    if i in Table1:
                        Head = Table1[i]
                        Tail = Table0[i]
                        s1 = P[s]
                        s2 = P[s+1]
                        Element = (s1,s2,s,s+1)
                        new_Arrow = (Head, Tail, Element, Sign)
                        NewArrowList.append(new_Arrow)

                    else : # We have a join map.
                        Head = Table2[i]
                        Tail = Table0[i]
                        new_Arrow1 = (Head  , Tail, ()     , Sign) # Maps (v+) to  1. 
                        new_Arrow2 = (Head+1, Tail, (s,s+1), Sign) # Maps (v-) t0  x.
                        NewArrowList+=[new_Arrow1, new_Arrow2]

        GeneratorList = NewGeneratorList
        ArrowList     = NewArrowList

    List1 = [(a,b) for  _, a, b in GeneratorList]
    List2 = [(Head,Tail,Sign) for Head, Tail, _ , Sign in ArrowList]
    
    return List1, List2               
       

 
