from knot_utilities import IsKnot, Writhe
from collections import defaultdict

########## KAUFFMAN BRACKET AND JONES POLYNOMIAL ##########

###### Functions:

## KauffmanBracket(K)
## Jones(K)
## PrintJones(K)


def KauffmanBracket(K):
    """
    For a knot diagram K it returns the Kauffman Bracket of $K$.

    Each crossing can be resolved in two ways (1 or -1) using the conventions: 
    For a right-handed twists:
    replacing the arcs by line segments paralel to the y axis is 1,
    replacing with a minimum and a maximum is -1.
    For left handed twists it is the opposite.

    For a projection with n crossings there are 2^n resolutions.  
    If r is one of the 2^n resolutions it gets two integers: 
    f(r) = (number of circles in r) -1
    g(r) = (number of 1 resolutions) - (number of -1 resolutions) 
    
    The Kauffman Bracket is defined by a state sum where the contribution of r is
    (-A^2-A^{-2})^{f(r)} \cdot A^{g(r)}
    However this is not practical where n is large. 
    
    The algorithm uses dynamic programming and computes a set of polynomials that solve the following problem: 
    
    Look at the part of the projection that lies above a horizontal y = C line with $k$ crossings.
    If we resolved the crossings that are above this line we would get a collection of arcs and circles
    for all the 2^k resolutions.
    If the line intersects the projection at 2t points, there are t arcs. 
    This information is encoded by P = (i_1,i_2,..., i_{2t}) where i and P[i] are connected by an arc.
    For each P we get a polynomial, where we only sum over resolutions with the topological type of P.

    The algorithm computes these polynomials by starting at 
    C = (global maximum of the projection) + epsilon 
    and decreasing C step by step to 
    C = global minimum - epsilon

    The answer is given as ((i_1,j_1,..., i_k,j_k)) where i_t < i_{t+1} and j is non-zero.
    
    The contribution of (i,j) is j\cdot A^i
    """

    if IsKnot(K) == False:
        print("Not a knot")
        return
    Dict1 = defaultdict(int)
    Dict1[((1,0),0)] =  1 # After passing through the unique global maximum.
    
    for i in range(1,len(K)-1):
        c = K[i]
        Dict2 = defaultdict(int)

        if c > 100: # Passing through a maximum.
            s = c-101
            for key in Dict1:
                value = Dict1[key]
                P, a = key
                new_P = [p if p < s else p+2 for p in P[:s]] + [s+1,s] + [p if p < s else p+2 for p in P[s:]]
                new_P = tuple(new_P)
                new_key = (new_P, a)
                Dict2[new_key] += value

        elif c < -100: # Passing through a minimum.
            s = -c -101
            for key in Dict1:
                value = Dict1[key]
                P, a = key
                s1 = P[s]
                s2 = P[s+1]
                if s1 != s+1:
                    new_P = list(P)
                    new_P[s1] = s2
                    new_P[s2] = s1
                    new_P = [p if p < s else p-2 for p in new_P[:s]] + [p if p < s else p-2 for p in new_P[s+2:]]
                    new_P = tuple(new_P)
                    new_key = (new_P, a)
                    Dict2[new_key]+= value
                   
                else : # we create a new circle, multiply with (-A^{-2}-A^2)
                    new_P = [p if p < s else p-2 for p in P[:s]] + [p if p < s else p-2 for p in P[s+2:]]
                    new_P = tuple(new_P)
                    new_key1 = (new_P, a+2)
                    new_key2 = (new_P, a-2)
                    Dict2[new_key1]-=value
                    Dict2[new_key2]-=value
            
        else :  # Passing through a crossing. 
            s = abs(c)-1
            # computing the convention for the resolutions:
            if c > 0:
                type_of_parallel_res = +1
                type_of_other_res    = -1
            else:
                type_of_parallel_res = -1
                type_of_other_res    = +1
            
            for key in Dict1:
                value = Dict1[key]
                P, a = key
                # First add the contribution of the resolution that is parallel to the y axis: 
                new_key = (P, a + type_of_parallel_res)
                Dict2[new_key] += value 

                # Then add the contribution of the other resolution:
                s1 = P[s]
                s2 = P[s+1]
                if s1 != s+1:
                    new_P = list(P)
                    new_P = new_P[:s] + [s+1,s] + new_P[s+2:]
                    new_P[s1] = s2
                    new_P[s2] = s1
                    new_P = tuple(new_P)
                    new_key = (new_P, a + type_of_other_res)
                    Dict2[new_key] +=value
                
                else:
                    # In this case s is paired with s+1.
                    # We add a circle, so we multiply with (-A^{-2}-A^2).
                    # The topological type of P is unchanged.
                    
                    new_key1 = (P, a + type_of_other_res + 2)
                    new_key2 = (P, a + type_of_other_res - 2)
                    Dict2[new_key1] -= value
                    Dict2[new_key2] -= value
        
        Dict1 = Dict2
        
    ans = []

    for key in Dict1:
        P, a = key
        b = Dict1[key]
        if b != 0:
            ans.append((a,b))
    ans.sort()

    return tuple(ans)
        

def Jones(K):
    """
    For a knot diagram K it returns the Jones polynomial of $K$.
      
    One of the definitions uses the Kauffman bracket:
    
    Take (-A)^{3\cdot Writhe(Projection)}\cdot(Kauffman(Projection)) 
    and substitute t^{1/2} =  A^{-2} to get Jones(K)(t).

    The Writhe(K) is the number of positive crossings minus the number of negative crossings.
    
    The function uses KauffmanBracket(K) and Writhe(K).
    """

    if IsKnot(K) == False:
        print("Not a knot")
        return 
    
    w = Writhe(K)
    L = KauffmanBracket(K)
    ans = []
    for a, b in L:
        # First we multiply the Kauffman bracket with (-A)^{3\cdot Writhe(Projection)}:
        if w%2 == 0:
            new_b = b
        else:
            new_b = -b
        new_a = a -3*w
        
        # Then we subsitute t^{1/2} = A^{-2}:
        new_a = (-new_a)//4 # Since K is a knot projection, new_a is divisible by 4.
        ans.append((new_a,new_b))
    ans.sort()

    return tuple(ans)


def PrintJones(K):
    """
    For a knot diagram K it returns the Jones polynomial of K in a string format.
    """
    
    if IsKnot(K) == False:
        print("Not a knot")
        return 
    L = Jones(K)
    ans = ''
    for a,b in L[:]:
        if b > 0:
            ans+='+'
        else:
            ans+='-'
        if (a, abs(b)) == (0,1):
            ans+='1'
                           
        if abs(b) > 1:
            ans+=str(abs(b))
        if a != 0:
            ans+='T^{'+str(a)+'}'
    if ans[0] == '+':
        ans = ans[1:]

    return ans

def CrossingBounds(K):
    """
    For a diagram K it returns some bounds for the crossing number.
    
    The lower bound is the span (or breadth) of the Jones polynomial:
    if Jones(K) = a_{i_1}T^{i_1}+ \cdots + a_{i_k}T^{i_k} 
    where i_t < i_t+1  and a_{i_1}, a_{i_k} are non-zero then
    breadth of Jones(K) = i_k-i_1

    It follows from the definition of the Jones polynomial 
    (using the Kauffman Bracket) that:  
    breadth of Jones(K) <= minimal crossing number of K. 

    The upper bound is just the number of crossings in K.
    
    The function returns [lower_bound, upper_bound]
    """

    if IsKnot(K) == False:
        print("Not a knot")
        return
    
    number_of_crossings = 0
    for c in K:
        if abs(c) < 100:
            number_of_crossings+=1
    upper_bound = number_of_crossings
    
    J = Jones(K)
    minimal_degree = J[0][0]
    maximal_degree = J[-1][0]
    lower_bound = maximal_degree - minimal_degree
    return [lower_bound, upper_bound]
    

        
