from collections import defaultdict
from knot_utilities import IsKnot, PositiveQuadrants
from alexander1 import Alexander

########## KNOT FLOER GENERATORS  ############

### Functions that are included:

## KnotFloerGenerators
## GenusBounds2


def KnotFloerGenerators(K):
    """
    For a knot diagram K returns the number of generators of the Knot Floer complex for each bigrading. 
    
    The return value is a tuple ((a_1,m_1,v_1),...,(a_n,m_n,v_n)), where v are non-zero.
    The ordering for (a_i,m_i) is lexicographical: a_t <= a_{t+1}, if a_t = a_{t+1} then m_t < m_{t+1}. 

    A term (a,m,v) means that there are v generators with Alexander grading = a and Maslov grading = m.

    The computation uses the same inductive scanning algorithm as the function Alexander(K).
    """

    if IsKnot(K) == False:
        print("Not a knot")
        return
    
    Dict_Crossing = PositiveQuadrants(K)
    # The orientation of strands at the crossings.
    # Example: If both strands are oriented from left to right, then the value is E (East).

    Dict1 = defaultdict(int)

    # Suppose the unique global maximum has y coordinate = 2*len(K) and maximums, minimums and crossings are at even y coordinates.
    # K[i] represents a maximum, minimum or crossing at y = 2*len(K)-2i.

    # Dict1 will record the contributions of upper Kauffman states that lie above a horizontal line y = C.
    # After step i the line is at y = 2*len(K)-1-2*i.

    # Dict1[(P,a,m)] is the number of generators with a given (P,a, m),
    # where a = 2*Alexander grading, m is the Maslov grading
    # and P is given by the list of regions that still require a Kauffman corner.

    # Strands that intersect the y = C line are counted from left to right: 1,2,...,2n.
    # P = (p_1,...,p_n) where 0 < p_1 < p_2 ... p_n < 2n.
    
    # If p \in P then the two dimensional region that lies below the y = C line
    # and borders the line-segment between strands p and p+1 still needs a Kauffman corner.

    Dict1[((1,), 0, 0)] = 1

    # After the global maximum we have two strands, 1 and 2.
    # The unique Kauffman state corresponds to the empty set.
    # The (Alexander, Maslov) bigrading is (0,0)
    # The region between strands 1 and 2 needs a Kauffman corner.

    for step in range(1,len(K)-1):
        Dict2 = defaultdict(int)
        c = K[step]
        if c > 100: # Adding a Maximum.
            s = c-101 # The maximum is between the newly created strands s+1 and s+2. Strands with value > s shift to value+2.
            for key in Dict1:
                value = Dict1[key]
                P, a, m = key
                new_P = [p if p < s else p+2 for p in P if p != s]+[s+1]
                if s not in P:
                    new_P = tuple(sorted(new_P))
                    Dict2[(new_P, a, m)] += value
                else :
                    # In this case the interval s was required to have a Kauffman corner below.
                    # After the maximum this interval splits into s and s+2.
                    # Exactly one of these intervals will require a Kauffman corner below.

                    new_P1 = tuple(sorted(new_P+[s]))
                    Dict2[(new_P1, a, m)] += value
                    new_P2 = tuple(sorted(new_P+[s+2]))
                    Dict2[(new_P2, a, m)] += value

        elif c < -100: # Adding a Minimum. 
            s = -c-101 # The minimum is between strands s+1 and s+2. Strands with value > s+2 shift to value-2.
            for key in Dict1:
                value  = Dict1[key]
                P, a, m = key
                if s+1 not in P: # If s+1 in P then we can't extend P.
                    new_P = [p if p < s else p-2 for p in P if p not in [s,s+2]]
                    if s in P and s+2 in P:
                        new_key = (tuple(sorted(new_P+[s])), a, m)
                        Dict2[new_key] += value
                    if (s in P and s+2 not in P) or (s not in P and s+2 in P):
                        new_key = (tuple(sorted(new_P)), a, m)
                        Dict2[new_key] += value

        else :
            s = abs(c) # Adding a crossing between strands s and s+1. If c > 0 it is right-handed, if c < 0 left-handed.
            Q = Dict_Crossing[step] # Describes the orientation of the crossing. Q is in [N, E, S, W].
            if Q == 'N':
                LocalA = [-1,0,1,0]
                LocalM = [-1,0,0,0]
            if Q == 'E':
                LocalA = [0,1,0,-1]
                LocalM = [0,1,0, 0]
            if Q == 'S':
                LocalA = [1,0,-1,0]
                LocalM = [0,0,-1,0]
            if Q == 'W':
                LocalA = [0,-1,0,1]
                LocalM = [0, 0,0,1]
            if c < 0:
                LocalA = [-z for z in LocalA]
                LocalM = [-z for z in LocalM]
           
            for key in Dict1:
                value = Dict1[key]
                P, a, m = key
                if s in P:  # Adding a North Corner:
                    new_a = a + LocalA[0]
                    new_m = m + LocalM[0]
                    new_key = (P, new_a, new_m)
                    Dict2[new_key]+= value
                        
                if s not in P and (s+1 in P): # Adding an East Corner
                    new_a = a + LocalA[1]
                    new_m = m + LocalM[1]
                    new_P = tuple(sorted([p if p != s+1 else s for p in P]))
                    new_key = (new_P, new_a, new_m)
                    Dict2[new_key]+= value
                
                if s not in P: # Adding a South Corner:
                    new_a = a + LocalA[2]
                    new_m = m + LocalM[2]
                    new_key = (P, new_a, new_m)
                    Dict2[new_key]+= value
                    
                if s not in P and (s-1 in P): # Adding a West Corner:
                    new_a = a + LocalA[3]
                    new_m = m + LocalM[3]
                    new_P = tuple(sorted([p if p != s-1 else s for p in P]))
                    new_key = (new_P, new_a, new_m)
                    Dict2[new_key]+= value
        Dict1 = Dict2

    # The last step adds the unique global minimum. We place the marked edge here.
    # For each (P, a) in Dict1 we have  P = (1,).
    # Because of the marked edge the region between strands 1 and 2 no longer requires a Kaufmann corner.
    
    ans = []
    for key in Dict1:
        _ , a, m = key
        value = Dict1[key]
        if value != 0:
            ans.append((a//2,m, value))
    ans.sort()
    ans = tuple(ans)  
    return ans

    
def GenusBounds2(K):
    """
    For a knot diagram K it returns a range for the possible values for the Seifert genus.
   
    Note that the Knot Floer complex can be used to compute the Seifert genus of K.
    However here we only use the Knot Floer generators.
 
    A simpler approach to get an upper bound is to use the Seifert algorithm and compute the genus of the resulting surface.
    (See the function GenusBounds in seifert.py)
      
    The upper bound for g(K) is the maximal Alexander grading among the Knot Floer generators.
    The lower bound is the degree of the symetrized Alexander polynomial.

    It returns [lower_bound, upper_bound].
    """

    if IsKnot(K) == False:
        print("Not a knot")
        return

    List1 = Alexander(K)
    List2 = KnotFloerGenerators(K)
    d, _ = List1[-1] # The maximal non-trivial term in the Alexander polynomial

    min_alexander_grading, _, _ = List2[0]
    max_alexander_grading, _, _ = List2[-1]

    # Using g(K) = g(Mirror(K)) and  the symmetry that relates the knot Floer generators of K, Mirror(K):

    best_upper_bound_from_the_projection = min(-min_alexander_grading, max_alexander_grading)
    g_min = d
    g_max = best_upper_bound_from_the_projection
    return [g_min, g_max]


    

 
 

    
