
############ UTILITY FUNCTIONS FOR PROJECTIONS ##################

#### Functions included:

## IsKnot
## PositiveQuadrants
## PosNegCrossings
## NPlusMinus
## Writhe
## BadCrossings
## IsAlternatingProjection


def IsKnot(K)-> bool:
    """
    Checks if K is a valid description of a knot as a sequence of maximums, minimums and crossings.
    K is a list (or tuple) of integers, with the following conventions:
    Maximums between strands t and t+1 are represented by  100+t.
    Minimums between strands t and t+1 are represented by -100+t.
    Right-handed twists between strands t and t+1 are represented by  t.
    Left-handed  twists between strands t and t+1 are represented by -t.
    
    Example: (101,-101) for the unknot.
    Example: (101, 103, 2, 2, 2, -101, -101) for the right-handed trefoil RHT.
    Example: (101, 102, 104, -1, -3, -5, -104, -102, -101) for RHT as the Pretzel knot P(-1,-1,-1).
    Example: (101, 103, 2, 2, 2, -101, 103, -2, -2, -2, -103, -101) for the connected sum RHT # LHT.
    K has to start with a maximum and has to end with a minimum, otherwise maximums and minimums can appear anywhere.
    Using the convention that the last minimum is oriented from left to right K represents an oriented knot.
    

    Returns False if it is not a valid projection of a link, or if the link has more than 1 component.
    """

    if  K == None or len(K) < 2:
        return False

    if K[0] != 101 or K[-1] != -101:
        return False

    number_of_maximums = 0
    number_of_minimums = 0
    for c in K:
        if c > 100:
            number_of_maximums+=1
        if c < -100:
            number_of_minimums+=1
    if number_of_maximums != number_of_minimums:
        return False


    n = len(K)
    M = [1,0]
 
    for i in range(1,n-1): 
        c = K[i]
        if c == 0 or c > len(M)+101 or c < -99-len(M) or (abs(c) <101 and abs(c)> len(M)-1):
            return False
        if c > 99:
            s = c-101
            M = [x if x < s else x+2 for x in M]
            M = M[:s]+[s+1,s]+M[s:]
        elif c < -99:
            s = -c-101
            a = M[s]
            b = M[s+1]
            if a == s+1 and b == s:
                return False
            M[a] = b
            M[b] = a
            M = M[:s]+M[s+2:]
            M = [x if x < s else x-2 for x in M]
        else :
            s = abs(c)-1
            a = M[s]
            b = M[s+1]
            if a != s+1:
                M[a] = s+1
                M[b] = s
                M[s] = b
                M[s+1] = a
    return True  


def PositiveQuadrants(K):
    """
    Given an oriented knot and a crossing, the positive quadrant is 
    bounded by the two line segments that originate from the crossing.
    
    For example if the crossing is between strands t and t+1  and 
    both are oriented upwards then this would be the North quadrant.
    If both strands are oriented from left to right 
    it would be the East quadrant.
    
    This utility function computes this info for each crossings.
    It returns a dictionary. 
    If K[i] is a crossing then value at i is either 'N', 'E', 'S' or 'W'.
    
    Example: for K = (101, 102, 104, 1, 1, 3, 5, -102, -102, -101) (4_1 knot as P(2,1,1))
    the four crossings are at 3<= i <= 6
    and we get {3: 'E', 4: 'W', 5: 'S', 6: 'N'}
    The algorithm starts at the last minimum and travels along the knot.
    When it encounters a crossing it records the direction it travels (Up, Down, Left, Right)
    """

    Dict = {} 
    if IsKnot(K) == False:
        print("Not a knot")
        return 

    t = len(K)
    number_of_crossings = 0 
    for x in K:
        if abs(x) < 100:
            number_of_crossings+=1 

    position = t-1
    Up = True
    strand = 1
  
    while len(Dict) < 2*number_of_crossings:
        if Up:
            position = position -1
        else:
            position = position +1
        if position < 0:
            break
        c = K[position]
        if Up:
            if c > 100:
                s = c-101
                
                if strand == s:
                    Up = False
                    strand = s+1
                elif strand == s+1:
                    Up = False
                    strand = s
                elif strand > s+1:
                    strand = strand-2
            elif c < -100:
                s = -c-101
                if strand > s-1:
                    strand = strand+2
            else:
                s = abs(c)-1
                if strand == s:
                    Dict[(position,'R')] ='Up'
                    strand = s+1
                elif strand == s+1:
                    Dict[(position, 'L')] ='Up'
                    strand = s
        else :
            if c  > 100:
                s = c-101
                if strand > s-1:
                    strand = strand + 2
            elif c < -100:
                s = -c-101
                if strand == s:
                    Up = True
                    strand = s+1
                elif strand == s+1:
                    Up = True
                    strand = s
                elif strand > s+1:
                    strand = strand-2
            else:
                s = abs(c)-1
                if strand == s:
                    Dict[(position, 'L')] = 'Down'
                    strand = s+1
                elif strand == s+1:
                    Dict[(position, 'R')] = 'Down'
                    strand = s
    Dict2 = {}
    for i in range(t):
        if (i, 'L') in Dict:
            a = Dict[(i,'L')]
            b = Dict[(i,'R')]
            if a == 'Up'  and b == 'Up':
                c = 'N'
            elif a == 'Up'  and b == 'Down':
                c = 'W'
            elif a =='Down' and b == 'Up':
                c = 'E'
            elif a =='Down' and b == 'Down':
                c = 'S'
            Dict2[i] = c
    return Dict2       
    

def PosNegCrossings(K):
    """
    For a knot diagram it returns the list of positive and negative crossings.
    """
    Dict = PositiveQuadrants(K)
    if Dict == None:
        return 
    Pos = []
    Neg = []
    for i in range(len(K)):
        c = K[i]
        if abs(c) < 100:
            v = Dict[i]
            if v =='S' or v == 'N':
                if c > 0:
                    Pos.append(i)
                else:
                    Neg.append(i)
            else:
                if c > 0:
                    Neg.append(i)
                else:
                    Pos.append(i)
    return (tuple(Pos), tuple(Neg))


def NPlusMinus(K):
    """
    For a knot diagram it returns (n+,n-) where
    n+ is the number of positive crossings in the diagram and
    n- is the number of negative crossings in the diagram.
    Uses the function PositiveQuadrants.
    Example: K = (101, 102, 104, 1, 1, 3, 5, -102, -102, -101)
    NPlusMinus(K) = (2, 2)
    Example K = (101, 102, 104, 1, 3, 5, -102, -102, -101)
    NPlusMinus(K) = (0, 3)
    """
    Cr = PosNegCrossings(K)
    if Cr == None:
        return
    Pos, Neg = Cr
    return (len(Pos), len(Neg))


def Writhe(K):
    """ 
    Returns the writhe of the projection.
    It is defined as the  number of positive crossings 
    minus the number of negative crossings.    
    """
    a = NPlusMinus(K)
    if a == None:
        return 

    n_plus, n_minus = a
    return n_plus-n_minus


def BadCrossings(K):
    """
    Given a projection K it returns a list of crossings in K so that if we change all of them
    the projection becomes alternating (of course this will likely change the knot). 
    
    Example: 
    For K = (101, 102, 104, 1, 1, 3, -5, -102, -102, -101) it will return [6]. 
    Here K[6]= -5 can to be changed to 5 to make it an alternating projection.
    
    Note that the original projection is Pretzel(2,1,-1) and represents the Unknot.
    The new projection is Pretzel(2,1,1) and represents the 4_1 knot. 

    There are always two solutions, the other answer is the complement.
    In the example [3,4,5] will change it to Pretzel(-2,-1,-1).
    The function will return an answer with the minimum number of crossings.  
    """

    Dict = {} 
    if IsKnot(K) == False:
        print("Not a knot")
        return 

    number_of_crossings = 0
    for i in K:
        if abs(i) < 100:
            number_of_crossings +=1

    position = len(K)-1
    Up = True
    strand = 1
    Over = True
    Good = []
    Bad  = []
  
    while len(Good)+len(Bad) < 2*number_of_crossings:
        if Up:
            position = position -1
        else:
            position = position +1
        if position < 0:
            break
        c = K[position]
        if Up:
            if c > 100:
                s = c-101
                
                if strand == s:
                    Up = False
                    strand = s+1
                elif strand == s+1:
                    Up = False
                    strand = s
                elif strand > s+1:
                    strand = strand-2
            elif c < -100:
                s = -c-101
                if strand > s-1:
                    strand = strand+2
            else:
                s = abs(c)-1
                if strand == s:
                    IsOver = True if c > 0 else False
                    strand = s+1
                    if IsOver == Over:
                        Good.append(position)
                    else:
                        Bad.append(position)
                    Over = (Over == False)
                elif strand == s+1:
                    IsOver = True if c <  0 else False
                    strand = s
                    if IsOver == Over:
                        Good.append(position)
                    else:
                        Bad.append(position)
                    Over = (Over == False)
               
                    
        else :
            if c  > 100:
                s = c-101
                if strand > s-1:
                    strand = strand + 2
            elif c < -100:
                s = -c-101
                if strand == s:
                    Up = True
                    strand = s+1
                elif strand == s+1:
                    Up = True
                    strand = s
                elif strand > s+1:
                    strand = strand-2
            else:
                s = abs(c)-1
                if strand == s:
                    IsOver = True if c < 0 else False
                    strand = s+1
                    if IsOver == Over:
                        Good.append(position)
                    else:
                        Bad.append(position)
                    Over = (Over == False)
               
                elif strand == s+1:
                    IsOver = True if c > 0 else False
                    strand = s
                    if IsOver == Over:
                        Good.append(position)
                    else:
                        Bad.append(position)
                    Over = (Over == False)
        
    if len(Bad) > len(Good):
        Bad = Good
    Bad.sort()
    Bad = [Bad[i] for i in range(0,len(Bad),2)] # Each bad crossing appears twice.
    return Bad  


def IsAlternatingProjection(K):
    if IsKnot(K) == False:
        print("Not a knot")
        return
    return len(BadCrossings(K)) == 0

