from alexander1 import KnotDeterminant
from knot_utilities import IsKnot, IsAlternatingProjection, PosNegCrossings, BadCrossings, NPlusMinus

########## KNOT SIGNATURE #########

### It contains the functions:

## KnotSignatureForAlternating
## KnotSignature

def KnotSignatureForAlternating(K):
    """
    For an alternating projection it computes the knot signature sigma(K).
    
    sigma(K) is defined as the signature of (S+S^T), where S is a Seifert matrix for K.

    The algorithm uses the Goeritz formula for alternating projections.
    This uses a chessboard coloring of the regions, where regions that meet at an edge are colored differently.
    In an alternating projection the local picture around each crossing is the same.
    """

    if IsKnot(K) == False:
        print("Not a knot")
        return
    if IsAlternatingProjection(K) == False:
        print("Not an alternating projection")
        return

    N_plus, N_minus = NPlusMinus(K)

    # For each upper region we associate: 1-(the number of intervals it meets on the y = C line).
    # Then we add this up for both colors.
    # At each step we calculate the change.
    
    Parity = 0

    # After passing through the unique global maximum:
    
    ColorA = 1 # The infinite region uses color A. It doesn't meet any of the finite intervals.
    ColorB = 0 # There is a small region that meets the interval between  strands 1 and 2, so it's contribution is 1 - 1.

    for step in range(1, len(K)):
        c = K[step]
        if abs(c) < 100 and ((c > 0 and c%2 == 1) or (c < 0 and c%2 == 0)):
            Parity = 1 # Since the projection is alternating, Parity has the same value at each crossing. 
        if c > 100:
            s = 101-c
            if s%2 == 0:
                ColorA -=1
            else:
                ColorB -=1
        if c < -100:
            s = -c-101
            if s%2 == 0:
                ColorB+=1
            else:
                ColorA+=1
        if abs(c) < 100:
            s = abs(c)
            if s%2 == 0:
                ColorA +=1
            else:
                ColorB +=1

    if Parity == 1:
        ColorA, ColorB = ColorB, ColorA

    # Note that ColorA + ColorB = Number_Of_Crossings + 2 = N_plus + N_Minus + 2
    # The Goeritz formula for the signature of alternating knots is
    # sigma = N_minus - ColorA + 1 = ColorB - N_plus -1

    return N_minus - ColorA + 1


def KnotSignature(K):
    """
    Returns the Knot signature sigma(K).

    It is defined as the signature of S+S^T, where S is a 2g by 2g matrix called the Seifert matrix for K.
    
    Note that sigma(K) is even for each knot K: 
    since Det(S+S^T) = Det(K) = Alexander(K)(-1) is odd it follows that S+S^T is non-singular.

    The algorithm uses the following result: 
    
    Change a crossing from positive to negative:
    if Det(K+) and Det(K-) have the same sign then  sigma(K+) = sigma(K-) 
    if Det(K+) and Det(K-) have opposite signs then sigma(K+) = sigma(K-)-2

    By changing some crossings in K we get to an alternating projection 
    and then use the KnotSignatureForAlternating function.
    """
    if IsKnot(K) == False:
        print("Not a knot")
        return 
    List1 = BadCrossings(K)
    Pos, Neg  = PosNegCrossings(K)
    new_K = list(K)
    old_Det = KnotDeterminant(K)
    Change = 0
    for c in List1:
        new_K[c] = -new_K[c]
        new_Det = KnotDeterminant(new_K)
        if old_Det*new_Det < 0:
            if c in Pos: # we changed a positive crossing to a negative
                Change+=2
            else:
                Change-=2
        old_Det = new_Det

        
    # Now we have an alternating knot new_K with 
    # sigma(new_K) = sigma(K) + Change

    return KnotSignatureForAlternating(new_K) - Change

        

    
     



