from collections import defaultdict
from knot_utilities import IsKnot, PositiveQuadrants

########## ALEXANDER POLYNOMIAL ############

### Includes the functions:
## Alexander
## PrintAlexander
## KnotDeterminant

def Alexander(K):
    """
    For a knot diagram K returns the symmetric Alexander polynomial of K.
    
    A_K(T) = \Sum _i (a_i)T^i, where a_i = a_{-i}, A_K(1) = 1.

    The return value is a tuple ((x_1,v_1),...,(x_n,v_n)), where v_t are non-zero and x_t  < x_{t+1).

    A term (x,v) contributes v\cdot(T^x) in the Alexander polynomial.

    The computation uses Kauffman states and an inductive scanning algorithm.
    """

    if IsKnot(K) == False:
        print("Not a knot")
        return
    
    Dict_Crossing = PositiveQuadrants(K)
    # The orientation of strands at the crossings.
    # Example: If both strands are oriented from left to right, then the value is E (East).

    Dict1 = defaultdict(int)

    # Suppose the unique global maximum has y coordinate = 2*len(K) and maximums, minimums and crossings are at even y coordinates.
    # K[i] represents a maximum, minimum or crossing at y = 2*len(K)-2i.

    # Dict1 will record the contributions of upper Kauffman states that lie above a horizontal line y = C.
    # After step i the line is at y = 2*len(K)-1-2*i.

    # Dict1[(P,a)] is the signed contribution of a given (P,a),
    # where a = 2*Alexander grading and P is given by the list of regions that still require a Kauffman corner.

    # Strands that intersect the y = C line are counted from left to right: 1,2,...,2n.
    # P = (p_1,...,p_n) where 0 < p_1 < p_2 ... p_n < 2n.
    
    # If p \in P then the two dimensional region that lies below the y = C line
    # and borders the line-segment between strands p and p+1 still needs a Kauffman corner.

    Dict1[((1,), 0)] = 1

    # After the global maximum we have two strands, 1 and 2.
    # The unique Kauffman state corresponds to the empty set.
    # The region between strands 1 and 2 needs a Kauffman corner.

    for step in range(1,len(K)-1):
        Dict2 = defaultdict(int)
        c = K[step]
        if c > 100: # Adding a Maximum.
            s = c-101 # The maximum is between the newly created strands s+1 and s+2. Strands with value > s shift to value+2.
            for key in Dict1:
                value = Dict1[key]
                P, a = key
                new_P = [p if p < s else p+2 for p in P if p != s]+[s+1]
                if s not in P:
                    new_P = tuple(sorted(new_P))
                    Dict2[(new_P, a)] += value
                else :
                    # In this case the interval s was required to have a Kauffman corner below.
                    # After the maximum this interval splits into s and s+2.
                    # Exactly one of these intervals will require a Kauffman corner below.
                    
                    new_P1 = tuple(sorted(new_P+[s]))
                    Dict2[(new_P1, a)] += value
                    new_P2 = tuple(sorted(new_P+[s+2]))
                    Dict2[(new_P2, a)] += value

        elif c < -100: # Adding a Minimum. 
            s = -c-101 # The minimum is between strands s+1 and s+2. Strands with value > s+2 shift to value-2.
            for key in Dict1:
                value  = Dict1[key]
                P, a = key
                if s+1 not in P: # If s+1 in P, then we can't extend P.
                    new_P = [p if p < s else p-2 for p in P if p not in [s,s+2]]
                    if s in P and s+2 in P:
                        new_key = (tuple(sorted(new_P+[s])), a)
                        Dict2[new_key] += value
                    if (s in P and s+2 not in P) or (s not in P and s+2 in P):
                        new_key = (tuple(sorted(new_P)), a)
                        Dict2[new_key] += value

        else :
            s = abs(c) # Adding a crossing between strands s and s+1. If c > 0 it is right-handed, if c < 0 left-handed.
            Q = Dict_Crossing[step] # Describes the orientation of the crossing. Q is in [N, E, S, W].
            if Q == 'N':
                LocalA = [-1,0,1,0]
                LocalM = [1,0,0,0]
            if Q == 'E':
                LocalA = [0,1,0,-1]
                LocalM = [0,1,0,0]
            if Q == 'S':
                LocalA = [1,0,-1,0]
                LocalM = [0,0, 1,0]
            if Q == 'W':
                LocalA = [0,-1,0,1]
                LocalM = [0,0,0,1]
            if c < 0:
                LocalA = [-z for z in LocalA]
           
            for key in Dict1:
                value = Dict1[key]
                P, a = key
                if s in P:  # Adding a North Corner:
                    new_a = a + LocalA[0]
                    Sign = 1
                    if LocalM[0] == 1: Sign = -1
                    new_key = (P, new_a)
                    Dict2[new_key]+= (Sign*value)
                        
                if s not in P and (s+1 in P): # Adding an East Corner
                    new_a = a + LocalA[1]
                    Sign = 1
                    if LocalM[1] == 1: Sign = -1
                    new_P = tuple(sorted([p if p != s+1 else s for p in P]))
                    new_key = (new_P, new_a)
                    Dict2[new_key]+= (Sign*value)
                
                if s not in P: # Adding a South Corner:
                    new_a = a + LocalA[2]
                    new_key = (P, new_a)
                    Sign = 1
                    if LocalM[2] == 1: Sign = -1
                    Dict2[new_key]+= (Sign*value)
                    
                if s not in P and (s-1 in P): # Adding a West Corner:
                    new_a = a + LocalA[3]
                    Sign = 1
                    if LocalM[3] == 1: Sign = -1
                    new_P = tuple(sorted([p if p != s-1 else s for p in P]))
                    new_key = (new_P, new_a)
                    Dict2[new_key]+= (Sign*value)
        Dict1 = Dict2

    # The last step adds the unique global minimum. We place the marked edge here.
    # For each (P, a) in Dict1 we have  P = (1,).
    # Because of the marked edge the region between strands 1 and 2 no longer requires a Kaufmann corner.
    
    ans = []
    for key in Dict1:
        _ , a = key
        value = Dict1[key]
        if value != 0:
            ans.append((a//2, value))
    ans.sort()
    ans = tuple(ans)  
    return ans

    
def PrintAlexander(K):
    """
    Returns the Alexander polynomial in a string format.
    """
    if IsKnot(K) == False:
        print("Not a knot")
        return 
    
    A = Alexander(K)
    ans = ''
    for a,b in A[:]:
        if b > 0:
            ans+='+'
        else:
            ans+='-'
        if (a, abs(b)) == (0,1):
            ans+='1'
                           
        if abs(b) > 1:
            ans+=str(abs(b))
        if a != 0:
            ans+='T^{'+str(a)+'}'
    if ans[0] == '+':
        ans = ans[1:]
    return ans

def KnotDeterminant(K):
    """
    For a knot diagram K returns det(K) = Alexander(K)(-1).
    
    Note that det(K)=Det(S+S^T) where S is the Seifert form of the knot.
    
    Since Alexander(K)(1) = 1, the det(K) for a knot is always an odd integer.
    """
    d = 0
    for a, b in Alexander(K):
        if a%2 == 0:
            d+=b
        else :
            d-=b
    return d

