from collections import defaultdict

############ EXAMPLES #################

def Unknot():
    return ((101, -101))

def RHT():
    return ((101,101,2,2,2,-101,-101))

def LHT():
    return ((101,101,-2,-2,-2,-101,-101))


def Pretzel(a,b,c):
    """
    Returns a diagram for the three stranded pretzel link P(a,b,c).
    Positive values indicate right-handed twists, negative values indicate left-handed twists.
    Example: Pretzel(1,1,1) The left handed trefoil LFT.
    Example: Pretzel(1,-1,1) the unknot.
    Example: Pretzel(-3,5,7). (A non-trivial knot with trivial Alexander polynomial).
    Note that P(a,b,c) is a knot if and only if at least two of a, b and c are odd.
    """

    K = [101,102,104]
    if a > 0:
        K+=([1]*a)
    if a < 0:
        K+=([-1]*abs(a))
    if b > 0:
        K+=([3]*b)
    if b < 0:
        K+=([-3]*abs(b))
    if c > 0:
        K+=([5]*c)
    if c < 0:
        K+=([-5]*abs(c))
    K+=[-102,-102,-101]
    return tuple(K)

def Torus(p,q):
    """
    Returns a diagram for the torus link T(p,q).
    Note that T(p,q)=T(q,p)=T(-p,-q), and T(p,-q) = Mirror(T(p,q).
    Note that T(p,q) is a knot if and only if (p,q)=1.
    """
    if min(abs(p), abs(q)) > 49:
        print("Too Large Values")
        return 
    t = p*q
    if t > 0: t =  1
    if t < 0: t = -1
    if t == 0:
        print("p and q are not allowed to be 0")
        return 0
    p = abs(p)
    q = abs(q) 
    if q < p:
        p, q = q, p
    K = []
    for i in range(p):
        K.append(100+i+1)
    for i in range(q):
        for j in range(p-1):
            K.append(t*(p-1-j))
    for i in range(p):
        K.append(-100-p+i)
    return tuple(K)

def Mirror(K):
    """ 
    Return a diagram for the Mirror of K.
    """
    return tuple([x if abs(x)> 100 else -x for x in K])

def Reverse(K):
    """
    Adds a Reidemeister 1 move at the last minimum.
    Since the knot is oriented from left to right at the last minimum, 
    the new projection represent the reverse of K.
    """
    return tuple(list(K[:-1])+[1, -101])

def Conway():
    """
    Returns a 12-crossing a presentation for the Conway knot.
    """
    K = (101,101,2,2,102,106,1,1,-3,-3,-3,-5,-5,7,7,7,-102, -102, -102,-101)
    return K


def KinoshitaTerasaka():
    """
    Returns a 12-crossing presentation for the Kinoshita-Terasaka knot.
    """
    K = (101,101,2,2,102,106,1,1,-3,-3,-3,5,5,5,-7,-7,-102, -102, -102,-101)
    return K


############ UTILITY FUNCTIONS FOR PROJECTIONS ##################


def IsKnot(K)-> bool:
    """
    Checks if K is a valid description of a knot as a sequence of maximums, minimums and crossings.
    K is a list (or tuple) of integers, with the following conventions:
    Maximums between strands t and t+1 are represented by  100+t.
    Minimums between strands t and t+1 are represented by -100+t.
    Right-handed twists between strands t and t+1 are represented by  t.
    Left-handed  twists between strands t and t+1 are represented by -t.
    
    Example: (101,-101) for the unknot.
    Example: (101, 103, 2, 2, 2, -101, -101) for the right-handed trefoil RHT.
    Example: (101, 102, 104, -1, -3, -5, -104, -102, -101) for RHT as the Pretzel knot P(-1,-1,-1).
    Example: (101, 103, 2, 2, 2, -101, 103, -2, -2, -2, -103, -101) for the connected sum RHT # LHT.
    K has to start with a maximum and has to end with a minimum, otherwise maximums and minimums can appear anywhere.
    Using the convention that the last minimum is oriented from left to right K represents an oriented knot.
    

    Returns False if it is not a valid projection of a link, or if the link has more than 1 component.
    """

    if  K == None or len(K) < 2:
        return False

    if K[0] != 101 or K[-1] != -101:
        return False

    number_of_maximums = 0
    number_of_minimums = 0
    for c in K:
        if c > 100:
            number_of_maximums+=1
        if c < -100:
            number_of_minimums+=1
    if number_of_maximums != number_of_minimums:
        return False


    n = len(K)
    M = [1,0]
 
    for i in range(1,n-1): 
        c = K[i]
        if c == 0 or c > len(M)+101 or c < -99-len(M) or (abs(c) <101 and abs(c)> len(M)-1):
            return False
        if c > 99:
            s = c-101
            M = [x if x < s else x+2 for x in M]
            M = M[:s]+[s+1,s]+M[s:]
        elif c < -99:
            s = -c-101
            a = M[s]
            b = M[s+1]
            if a == s+1 and b == s:
                return False
            M[a] = b
            M[b] = a
            M = M[:s]+M[s+2:]
            M = [x if x < s else x-2 for x in M]
        else :
            s = abs(c)-1
            a = M[s]
            b = M[s+1]
            if a != s+1:
                M[a] = s+1
                M[b] = s
                M[s] = b
                M[s+1] = a
    return True  



def GetPositiveQuadrants(K):
    """
    Given an oriented knot and a crossing, the positive quadrant is 
    bounded by the two line segments that originate from the crossing.
    
    For example if the crossing is between strands t and t+1  and 
    both are oriented upwards then this would be the North quadrant.
    If both strands are oriented from left to right 
    it would be the East quadrant.
    
    This utility function computes this info for each crossings.
    It returns a dictionary. 
    If K[i] is a crossing then value at i is either 'N', 'E', 'S' or 'W'.
    
    Example: for K = (101, 102, 104, 1, 1, 3, 5, -102, -102, -101) (4_1 knot as P(2,1,1))
    the four crossings are at 3<= i <= 6
    and we get {3: 'E', 4: 'W', 5: 'S', 6: 'N'}
    The algorithm starts at the last minimum and travels along the knot.
    When it encounters a crossing it records the direction it travels (Up, Down, Left, Right)
    """

    Dict = {} 
    if IsKnot(K) == False:
        print("Not a knot")
        return 

    t = len(K)
    number_of_crossings = 0 
    for x in K:
        if abs(x) < 99:
            number_of_crossings+=1 

    position = t-1
    Up = True
    strand = 1
  
    while len(Dict) < 2*number_of_crossings:
        if Up:
            position = position -1
        else:
            position = position +1
        if position < 0:
            break
        c = K[position]
        if Up:
            if c > 100:
                s = c-101
                
                if strand == s:
                    Up = False
                    strand = s+1
                elif strand == s+1:
                    Up = False
                    strand = s
                elif strand > s+1:
                    strand = strand-2
            elif c < -100:
                s = -c-101
                if strand > s-1:
                    strand = strand+2
            else:
                s = abs(c)-1
                if strand == s:
                    Dict[(position,'R')] ='Up'
                    strand = s+1
                elif strand == s+1:
                    Dict[(position, 'L')] ='Up'
                    strand = s
        else :
            if c  > 100:
                s = c-101
                if strand > s-1:
                    strand = strand + 2
            elif c < -100:
                s = -c-101
                if strand == s:
                    Up = True
                    strand = s+1
                elif strand == s+1:
                    Up = True
                    strand = s
                elif strand > s+1:
                    strand = strand-2
            else:
                s = abs(c)-1
                if strand == s:
                    Dict[(position, 'L')] = 'Down'
                    strand = s+1
                elif strand == s+1:
                    Dict[(position, 'R')] = 'Down'
                    strand = s
    Dict2 = {}
    for i in range(t):
        if (i, 'L') in Dict:
            a = Dict[(i,'L')]
            b = Dict[(i,'R')]
            if a == 'Up'  and b == 'Up':
                c = 'N'
            elif a == 'Up'  and b == 'Down':
                c = 'W'
            elif a =='Down' and b == 'Up':
                c = 'E'
            elif a =='Down' and b == 'Down':
                c = 'S'
            Dict2[i] = c
    return Dict2       
    

def GetPosNegCrossings(K):
    """
    For a knot diagram it returns (n+,n-) where
    n+ is the number of positive crossings in the diagram and
    n- is the number of negative crossings in the diagram.
    Uses the function GetPositiveQuadrants.
    Example: K = (101, 102, 104, 1, 1, 3, 5, -102, -102, -101)
    GetPosNegCrossings(K) = (2,2)
    Example K = (101, 102, 104, 1, 3, 5, -102, -102, -101)
    GetPosNegCrossings(K) = (0, 3)
    """
    if K == None:
        return
    t = len(K)
    Dict = GetPositiveQuadrants(K)
    if Dict == None:
        return 
    n_plus  = 0
    n_minus = 0
    for i in range(t):
        c = K[i]
        if abs(c) < 100:
            v = Dict[i]
            if v =='S' or v == 'N':
                if c > 0:
                    n_plus+=1
                else:
                    n_minus+=1
            else:
                if c > 0:
                    n_minus+=1
                else:
                    n_plus +=1
    return (n_plus, n_minus)

def Writhe(K):
    """ 
    Returns the writhe of the projection.
    It is defined as the  number of positive crossings 
    minus the number of negative crossings.    
    """
    a = GetPosNegCrossings(K)
    if a == None:
        return 

    n_plus, n_minus = a
    return n_plus-n_minus

    

########## KAUFFMAN STATES AND THE ALEXANDER POLYNOMIAL ############


def NumberOfKauffmanStates(K):
    if IsKnot(K) != True:
        print("Not a knot")
        return 
    
    Dict1 = defaultdict(int)
    Dict1[(1,)] = 1

    for i in range(1,len(K)-1):
        Dict2 = defaultdict(int)
        c = K[i]
        
        if c > 100: # Adding a Maximum
            s = c-101
            for P in Dict1:
                value = Dict1[P]
                new_P = [p if p < s else p+2 for p in P if p != s]+[s+1]
                if s not in P:
                    new_P = tuple(sorted(new_P))
                    Dict2[new_P]+=value
                else :
                    new_P1 = tuple(sorted(new_P+[s]))
                    Dict2[new_P1]+= value
                    new_P2 = tuple(sorted(new_P+[s+2]))
                    Dict2[new_P2]+= value

        elif c < -100: # Adding a Minimum
            s = -c-101
            for P in Dict1:
                value  = Dict1[P]
                if s+1 not in P:
                    new_P = [p if p < s else p-2 for p in P if p not in [s,s+2]]
                    if s in P and s+2 in P:
                        new_P = tuple(sorted(new_P+[s]))
                        Dict2[new_P] += value
                    if (s in P and s+2 not in P) or (s not in P and s+2 in P):
                        new_P = tuple(sorted(new_P))
                        Dict2[new_P] += value

        else:
            s = abs(c)
            for P in Dict1:
                value = Dict1[P]
                
                # Adding a North or South Corner:
                Dict2[P]+=value

                # Adding an East Corner:
                if s not in P and (s+1 in P): 
                    new_P = tuple(sorted([p if p != s+1 else s for p in P]))
                    Dict2[new_P]+=value
                    
                # Adding a West Corner:
                if s not in P and (s-1 in P): 
                    new_P = tuple(sorted([p if p != s-1 else s for p in P]))
                    Dict2[new_P]+=value
        Dict1 = Dict2
       

    ans = 0
    for key in Dict1:
        ans+=Dict1[key]   
    return ans

    

def Alexander(K):
    if IsKnot(K) == False:
        print("Not a knot")
        return 
    Dict_Crossing = GetPositiveQuadrants(K)
    Dict1 = defaultdict(int)
    Dict1[((1,), 0)] = 1
    # The signed contribution of a given (p,a),
    # where P = Position (idempotent), a = 2*Alexander grading
    for i in range(1,len(K)-1):
        Dict2 = defaultdict(int)
        c = K[i]
        if c > 100: # Adding a Maximum
            s = c-101
            for key in Dict1:
                value = Dict1[key]
                P, a = key
                new_P = [p if p < s else p+2 for p in P if p != s]+[s+1]
                if s not in P:
                    new_P = tuple(sorted(new_P))
                    Dict2[(new_P, a)] += value
                else :
                    new_P1 = tuple(sorted(new_P+[s]))
                    Dict2[(new_P1, a)] += value
                    new_P2 = tuple(sorted(new_P+[s+2]))
                    Dict2[(new_P2, a)] += value

        elif c < -100: # Adding a Minimum
            s = -c-101
            for key in Dict1:
                value  = Dict1[key]
                P, a = key
                if s+1 not in P:
                    new_P = [p if p < s else p-2 for p in P if p not in [s,s+2]]
                    if s in P and s+2 in P:
                        new_key = (tuple(sorted(new_P+[s])), a)
                        Dict2[new_key] += value
                    if (s in P and s+2 not in P) or (s not in P and s+2 in P):
                        new_key = (tuple(sorted(new_P)), a)
                        Dict2[new_key] += value

        else :
            s = abs(c)
            Q = Dict_Crossing[i] #Describes the orientation of the crossing: N, E, S, W
            if Q == 'N':
                LocalA = [-1,0,1,0]
                LocalM = [1,0,0,0]
            if Q == 'E':
                LocalA = [0,1,0,-1]
                LocalM = [0,1,0,0]
            if Q == 'S':
                LocalA = [1,0,-1,0]
                LocalM = [0,0, 1,0]
            if Q == 'W':
                LocalA = [0,-1,0,1]
                LocalM = [0,0,0,1]
            if c < 0:
                LocalA = [-z for z in LocalA]
           
            for key in Dict1:
                value = Dict1[key]
                P, a = key
                if s in P:  # Adding a North Corner
                    new_a = a + LocalA[0]
                    Sign = 1
                    if LocalM[0] == 1: Sign = -1
                    new_key = (P, new_a)
                    Dict2[new_key]+= (Sign*value)
                        
                if s not in P and (s+1 in P): # Adding an East Corner
                    new_a = a + LocalA[1]
                    Sign = 1
                    if LocalM[1] == 1: Sign = -1
                    new_P = tuple(sorted([p if p != s+1 else s for p in P]))
                    new_key = (new_P, new_a)
                    Dict2[new_key]+= (Sign*value)
                
                if s not in P: # Adding a South Corner
                    new_a = a + LocalA[2]
                    new_key = (P, new_a)
                    Sign = 1
                    if LocalM[2] == 1: Sign = -1
                    Dict2[new_key]+= (Sign*value)
                    
                if s not in P and (s-1 in P): # Adding a West Corner
                    new_a = a + LocalA[3]
                    Sign = 1
                    if LocalM[3] == 1: Sign = -1
                    new_P = tuple(sorted([p if p != s-1 else s for p in P]))
                    new_key = (new_P, new_a)
                    Dict2[new_key]+= (Sign*value)
        Dict1 = Dict2
       

    ans = []
    for key in Dict1:
        _ , a = key 
        value = Dict1[key]
        if value != 0:
            ans.append((a//2, value))
    ans.sort()
    ans = tuple(ans)  
    return ans

    
def PrintAlexander(K):
    if IsKnot(K) == False:
        print("Not a knot")
        return 
    
    A = Alexander(K)
    ans = ''
    for a,b in A[:]:
        if b > 0:
            ans+='+'
        else:
            ans+='-'
        if (a, abs(b)) == (0,1):
            ans+='1'
                           
        if abs(b) > 1:
            ans+=str(abs(b))
        if a != 0:
            ans+='T^{'+str(a)+'}'
    if ans[0] == '+':
        ans = ans[1:]
    return ans





